'use client';

import { useState, useEffect } from 'react';
import { Sparkles, TrendingUp, Globe, ArrowRight, Play, Award, Users } from 'lucide-react';
import { motion } from 'framer-motion';
import { useLanguage } from '@/contexts/language-context';
import Link from 'next/link';

type Particle = {
  id: number;
  left: string;
  top: string;
  duration: number;
  delay: number;
  size: number;
};

export default function HeroSection() {
  const { content, language } = useLanguage();
  const { hero } = content;
  const [particles, setParticles] = useState<Particle[]>([]);
  const [currentSlide, setCurrentSlide] = useState(0);
  const [mousePosition, setMousePosition] = useState({ x: 0, y: 0 });

  const heroImages = [
    { 
      id: 1, 
      color: 'from-blue-950/95 via-blue-900/90 to-blue-800/85',
      image: 'https://images.unsplash.com/photo-1497435334941-8c899ee9e8e9?w=1920&q=80',
      alt: 'Oil and Gas Energy Infrastructure'
    },
    { 
      id: 2, 
      color: 'from-emerald-900/90 via-green-800/85 to-teal-900/80',
      image: 'https://images.unsplash.com/photo-1504328345606-18bbc8c9d7d1?w=1920&q=80',
      alt: 'Mining and Industrial Operations'
    },
    { 
      id: 3, 
      color: 'from-slate-900/95 via-blue-950/90 to-emerald-900/85',
      image: 'https://images.unsplash.com/photo-1454165804606-c3d57bc86b40?w=1920&q=80',
      alt: 'Finance and Business Infrastructure'
    }
  ];

  useEffect(() => {
    const generateParticles = () => {
      if (typeof window !== 'undefined') {
        return [...Array(15)].map((_, i) => ({
          id: i,
          left: `${Math.random() * 100}%`,
          top: `${Math.random() * 100}%`,
          duration: 4 + Math.random() * 3,
          delay: Math.random() * 3,
          size: Math.random() * 2 + 1,
        }));
      }
      return [];
    };
    setParticles(generateParticles());

    const interval = setInterval(() => {
      setCurrentSlide((prev) => (prev + 1) % heroImages.length);
    }, 6000);

    return () => clearInterval(interval);
  }, []);


  useEffect(() => {
    const handleMouseMove = (e: MouseEvent) => {
      setMousePosition({
        x: (e.clientX / window.innerWidth - 0.5) * 20,
        y: (e.clientY / window.innerHeight - 0.5) * 20
      });
    };
    window.addEventListener('mousemove', handleMouseMove);
    return () => window.removeEventListener('mousemove', handleMouseMove);
  }, []);

  const features = [
    { icon: Sparkles, text: hero.features.innovative, color: 'from-amber-400 to-yellow-600' },
    { icon: TrendingUp, text: hero.features.sustainable, color: 'from-emerald-400 to-green-600' },
    { icon: Globe, text: hero.features.international, color: 'from-blue-500 to-blue-700' },
  ];

  return (
    <section className="relative h-screen min-h-[900px] w-full overflow-hidden pt-20">
      {/* Animated Background Images with Carousel */}
      <div className="absolute inset-0">
        {heroImages.map((img, index) => (
          <div
            key={img.id}
            className={`absolute inset-0 transition-opacity duration-2000 ${
              currentSlide === index ? 'opacity-100' : 'opacity-0'
            }`}
          >
            {/* Background Image */}
            <div 
              className="absolute inset-0 bg-cover bg-center transition-transform ease-out duration-[8000ms]"
              style={{
                backgroundImage: `url(${img.image})`,
                transform: currentSlide === index ? 'scale(1.1)' : 'scale(1)',
              }}
            />
            
            {/* Gradient Overlay */}
            <div className={`absolute inset-0 bg-gradient-to-br ${img.color}`} />
            
            {/* Additional Dark Overlay for Text Contrast */}
            <div className="absolute inset-0 bg-black/30" />
          </div>
        ))}
      </div>

      {/* Mesh Gradient Overlay */}
      <div className="absolute inset-0 opacity-30">
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_20%_30%,rgba(29,78,216,0.4),transparent_50%)]" />
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_80%_70%,rgba(217,119,6,0.3),transparent_50%)]" />
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_50%_50%,rgba(16,185,129,0.3),transparent_70%)]" />
      </div>

      {/* Animated Grid Pattern */}
      <div className="absolute inset-0 opacity-20">
        <div className="absolute inset-0" style={{
          backgroundImage: `linear-gradient(rgba(255,255,255,0.1) 1px, transparent 1px),
                           linear-gradient(90deg, rgba(255,255,255,0.1) 1px, transparent 1px)`,
          backgroundSize: '50px 50px',
          transform: `translate(${mousePosition.x * 0.2}px, ${mousePosition.y * 0.2}px)`
        }} />
      </div>

      {/* Floating Orbs */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute top-20 left-[10%] w-96 h-96 bg-gradient-to-br from-blue-600/30 to-blue-800/20 rounded-full blur-[100px] animate-pulse" 
             style={{ animationDuration: '4s' }} />
        <div className="absolute bottom-32 right-[15%] w-80 h-80 bg-gradient-to-br from-amber-500/30 to-yellow-600/20 rounded-full blur-[90px] animate-pulse" 
             style={{ animationDuration: '5s', animationDelay: '1s' }} />
        <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[600px] h-[600px] bg-gradient-to-br from-emerald-500/20 to-green-600/10 rounded-full blur-[120px] animate-pulse"
             style={{ animationDuration: '6s', animationDelay: '2s' }} />
      </div>

      {/* Particle System */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        {particles.map((particle) => (
          <motion.div
            key={particle.id}
            className="absolute rounded-full bg-white/60"
            style={{
              left: particle.left,
              top: particle.top,
              width: `${particle.size}px`,
              height: `${particle.size}px`,
              boxShadow: '0 0 10px rgba(255,255,255,0.5)'
            }}
            animate={{
                y: [0, -20, 0],
                x: [0, 10, 0]
            }}
            transition={{
                duration: particle.duration,
                delay: particle.delay,
                repeat: Infinity,
                ease: "easeInOut"
            }}
          />
        ))}
      </div>

      {/* Main Content */}
      <div className="relative z-10 flex h-full flex-col items-center justify-center text-center px-4">
        <div className="container max-w-7xl">
          
          {/* Premium Badge */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.1 }} 
            className="inline-flex items-center gap-3 rounded-full bg-white/10 backdrop-blur-xl px-6 py-3 border border-amber-400/30 shadow-2xl mb-8 group hover:bg-white/15 transition-all duration-300">
            <div className="relative">
              <Sparkles className="w-5 h-5 text-amber-400 animate-pulse" />
              <div className="absolute inset-0 bg-amber-400/20 rounded-full blur-md" />
            </div>
            <span className="text-sm font-bold text-white tracking-wide">{hero.badge}</span>
          </motion.div>

          {/* Main Title with Gradient Text */}
           <motion.h1 
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.2 }} 
            className="font-bold tracking-tight mb-6">
            <span className="block text-4xl sm:text-5xl lg:text-6xl mb-4 bg-gradient-to-r from-white via-blue-100 to-blue-200 bg-clip-text text-transparent drop-shadow-2xl animate-gradient">
              {hero.title}
            </span>
            <span className="block text-2xl sm:text-3xl lg:text-4xl bg-gradient-to-r from-amber-400 via-yellow-500 to-amber-600 bg-clip-text text-transparent font-normal drop-shadow-xl">
              {hero.title_line2}
            </span>
          </motion.h1>

          {/* Decorative Line */}
          <motion.div 
            initial={{ opacity: 0, scaleX: 0 }}
            animate={{ opacity: 1, scaleX: 1 }}
            transition={{ delay: 0.3 }}
            className="flex items-center justify-center gap-4 mb-8">
            <div className="h-[2px] w-20 bg-gradient-to-r from-transparent via-white/50 to-white/50 rounded-full" />
            <div className="w-2 h-2 rounded-full bg-white/70 shadow-lg shadow-white/50" />
            <div className="h-[2px] w-20 bg-gradient-to-l from-transparent via-white/50 to-white/50 rounded-full" />
          </motion.div>

          {/* Subtitle with Enhanced Typography */}
          <motion.p 
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.4 }}
            className="mx-auto max-w-3xl text-lg sm:text-xl lg:text-2xl text-white/90 leading-relaxed mb-12 font-light">
            {hero.subtitle}
          </motion.p>

          {/* Features Pills */}
          <motion.div 
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.5 }}
            className="flex flex-wrap justify-center gap-4 mb-12">
            {features.map((feature, index) => (
              <div
                key={index}
                className="group relative overflow-hidden rounded-2xl bg-white/10 backdrop-blur-xl px-6 py-3 border border-white/20 hover:border-white/40 transition-all duration-500 hover:scale-105 hover:shadow-2xl"
                style={{ animationDelay: `${index * 0.1}s` }}
              >
                <div className={`absolute inset-0 bg-gradient-to-r ${feature.color} opacity-0 group-hover:opacity-20 transition-opacity duration-500`} />
                <div className="relative flex items-center gap-3">
                  <div className={`p-2 rounded-lg bg-gradient-to-br ${feature.color} shadow-lg`}>
                    <feature.icon className="w-5 h-5 text-white" />
                  </div>
                  <span className="text-sm font-bold text-white">{feature.text}</span>
                </div>
              </div>
            ))}
          </motion.div>

          {/* CTA Buttons */}
          <motion.div 
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.6 }}
            className="flex flex-col sm:flex-row items-center justify-center gap-4 mb-16">
            <Link href="/services">
              <button className="group relative overflow-hidden rounded-2xl bg-gradient-to-r from-amber-500 via-yellow-600 to-amber-600 px-8 py-4 text-white font-bold shadow-2xl shadow-amber-500/50 hover:shadow-amber-500/70 transition-all duration-300 hover:scale-105">
                <div className="absolute inset-0 bg-gradient-to-r from-yellow-600 via-amber-600 to-yellow-500 opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
                <span className="relative flex items-center gap-2">
                  {hero.servicesButton}
                  <ArrowRight className="w-5 h-5 group-hover:translate-x-1 transition-transform" />
                </span>
              </button>
            </Link>
            
            <Link href="/contact">
              <button className="group relative overflow-hidden rounded-2xl bg-white/10 backdrop-blur-xl px-8 py-4 text-white font-bold border-2 border-emerald-400/50 hover:bg-emerald-500 hover:border-emerald-500 transition-all duration-300 hover:scale-105 shadow-xl">
                <span className="relative flex items-center gap-2">
                  <Play className="w-5 h-5" />
                  {hero.contactButton}
                </span>
              </button>
            </Link>
          </motion.div>

          {/* Scroll Indicator */}
          <div className="absolute bottom-12 left-1/2 -translate-x-1/2">
            <div className="flex flex-col items-center gap-2 animate-bounce">
              <div className="w-6 h-10 border-2 border-white/40 rounded-full flex items-start justify-center p-2">
                <div className="w-1.5 h-1.5 bg-white rounded-full animate-scroll" />
              </div>
              <span className="text-xs text-white/60 font-medium">{hero.scrollIndicator}</span>
            </div>
          </div>

          {/* Carousel Indicators */}
          <div className="absolute bottom-24 left-1/2 -translate-x-1/2 flex gap-2">
            {heroImages.map((_, index) => (
              <button
                key={index}
                onClick={() => setCurrentSlide(index)}
                className={`transition-all duration-300 rounded-full ${
                  currentSlide === index 
                    ? 'w-8 h-2 bg-amber-400' 
                    : 'w-2 h-2 bg-white/40 hover:bg-white/60'
                }`}
                aria-label={`Go to slide ${index + 1}`}
              />
            ))}
          </div>
        </div>
      </div>

      {/* Bottom Gradient Shadow */}
      <div className="absolute bottom-0 left-0 right-0 h-48 bg-gradient-to-t from-black/60 via-black/20 to-transparent pointer-events-none" />

      {/* CSS Animations */}
      <style jsx>{`
        @keyframes float {
          0%, 100% { transform: translateY(0px) translateX(0px); }
          25% { transform: translateY(-20px) translateX(10px); }
          50% { transform: translateY(-40px) translateX(-10px); }
          75% { transform: translateY(-20px) translateX(5px); }
        }
        
        @keyframes scroll {
          0% { transform: translateY(0); opacity: 0; }
          50% { opacity: 1; }
          100% { transform: translateY(16px); opacity: 0; }
        }
        
        @keyframes gradient {
          0% { background-position: 0% 50%; }
          50% { background-position: 100% 50%; }
          100% { background-position: 0% 50%; }
        }
        
        .animate-gradient {
          background-size: 200% 200%;
          animation: gradient 3s ease infinite;
        }
        
        .animate-scroll {
          animation: scroll 2s ease-in-out infinite;
        }
      `}</style>
    </section>
  );
}

    